<?php
declare(strict_types=1);

require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../inc/util.php';
require_admin();

$notice = '';
$error = '';

$uploadDirAbs = realpath(__DIR__ . '/..') . '/uploads';
$uploadDirRel = 'uploads';
if (!is_dir($uploadDirAbs)) @mkdir($uploadDirAbs, 0755, true);

function extract_paren_url(string $s): ?string {
  if (preg_match('/\((https?:\/\/[^)]+)\)/', $s, $m)) return $m[1];
  if (filter_var($s, FILTER_VALIDATE_URL)) return $s;
  return null;
}

function safe_filename(string $s): string {
  $s = preg_replace('/[^a-zA-Z0-9_\-\.]+/', '_', $s);
  return trim($s, '_');
}

function download_image(string $url, string $destAbs): bool {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 20,
    CURLOPT_CONNECTTIMEOUT => 6,
    CURLOPT_USERAGENT => 'BreakerCrossRefImporter/1.0',
  ]);
  $data = curl_exec($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  if ($data === false || $code >= 400) return false;
  @file_put_contents($destAbs, $data);
  return file_exists($destAbs) && filesize($destAbs) > 0;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_check((string)($_POST['csrf'] ?? ''))) {
    http_response_code(400);
    die('Bad CSRF token');
  }

  if (empty($_FILES['csv']['tmp_name']) || !is_uploaded_file($_FILES['csv']['tmp_name'])) {
    $error = 'Upload failed.';
  } else {
    $tmp = $_FILES['csv']['tmp_name'];
    $fh = fopen($tmp, 'r');
    if (!$fh) {
      $error = 'Could not read CSV.';
    } else {
      $headers = fgetcsv($fh);
      if (!$headers) {
        $error = 'CSV missing header row.';
      } else {
        $map = array_flip($headers);

        // Required columns (case sensitive to match your export)
        $required = ['Bin Number','Brand'];
        foreach ($required as $col) {
          if (!isset($map[$col])) {
            $error = 'Missing required column: ' . $col;
            break;
          }
        }

        if (!$error) {
          $pdo = db();
          $pdo->beginTransaction();

          // Upsert based on SKU if present, otherwise product_link, otherwise bin+brand+amp+poles+safety
          // Note: schema includes UNIQUE on SKU. If SKU is blank, the row will insert a new row.
          $stmt = $pdo->prepare("
            INSERT INTO breakers
              (bin_number, sku, friendly_sku, brand, amperage, poles, safety_features, mounting, image_path, product_link, is_active)
            VALUES
              (:bin_number, :sku, :friendly_sku, :brand, :amperage, :poles, :safety_features, :mounting, :image_path, :product_link, :is_active)
            ON DUPLICATE KEY UPDATE
              bin_number=VALUES(bin_number),
              friendly_sku=VALUES(friendly_sku),
              brand=VALUES(brand),
              amperage=VALUES(amperage),
              poles=VALUES(poles),
              safety_features=VALUES(safety_features),
              mounting=VALUES(mounting),
              image_path=COALESCE(VALUES(image_path), image_path),
              product_link=VALUES(product_link),
              is_active=VALUES(is_active)
          ");

          $count = 0;
          $downloaded = 0;
          $failedImg = 0;

          while (($row = fgetcsv($fh)) !== false) {
            $bin = trim((string)($row[$map['Bin Number']] ?? ''));
            $brand = trim((string)($row[$map['Brand']] ?? ''));

            if ($bin === '' || $brand === '') continue;

            $sku = trim((string)($row[$map['SKU']] ?? '')) ?: null;
            $friendly = trim((string)($row[$map['Friendly SKU']] ?? '')) ?: null;
            $amperage = trim((string)($row[$map['Amperage']] ?? '')) ?: null;
            $poles = trim((string)($row[$map['Poles']] ?? '')) ?: null;
            $safety = trim((string)($row[$map['Safety Features']] ?? '')) ?: null;
            $mounting = trim((string)($row[$map['Mounting']] ?? '')) ?: null;
            $link = trim((string)($row[$map['Link']] ?? '')) ?: null;

            // Active: optional column; default active
            $isActive = 1;
            if (isset($map['Active'])) {
              $isActive = ((string)($row[$map['Active']] ?? '1') === '0') ? 0 : 1;
            }

            // Image: may be "name (url)" from Airtable; download to local
            $imageCell = trim((string)($row[$map['Image']] ?? ''));
            $imgUrl = $imageCell ? extract_paren_url($imageCell) : null;

            $imagePathRel = null;

            // If they already provide a local path like uploads/..., keep it
            if ($imageCell && str_starts_with($imageCell, 'uploads/')) {
              $imagePathRel = $imageCell;
            } elseif ($imgUrl) {
              $baseName = safe_filename(($brand ?: 'brand') . '_' . ($friendly ?: ($sku ?: $bin)));
              $ext = 'jpg';
              if (preg_match('/\.(png|jpg|jpeg|webp|gif)\b/i', $imgUrl, $m)) $ext = strtolower($m[1]);
              $fileRel = $uploadDirRel . '/' . $baseName . '.' . $ext;
              $fileAbs = $uploadDirAbs . '/' . $baseName . '.' . $ext;

              if (!file_exists($fileAbs)) {
                if (download_image($imgUrl, $fileAbs)) {
                  $imagePathRel = $fileRel;
                  $downloaded++;
                } else {
                  $failedImg++;
                }
              } else {
                $imagePathRel = $fileRel;
              }
            }

            $stmt->execute([
              ':bin_number' => $bin,
              ':sku' => $sku,
              ':friendly_sku' => $friendly,
              ':brand' => $brand,
              ':amperage' => $amperage,
              ':poles' => $poles,
              ':safety_features' => $safety,
              ':mounting' => $mounting,
              ':image_path' => $imagePathRel,
              ':product_link' => $link,
              ':is_active' => $isActive,
            ]);

            $count++;
          }

          fclose($fh);
          $pdo->commit();

          $notice = "Imported/updated {$count} row(s). Downloaded {$downloaded} image(s). Failed image downloads: {$failedImg}.";
        }
      }
    }
  }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Batch Import</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
  <div class="container py-4" style="max-width: 900px;">
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h1 class="h4 m-0">Batch Import (CSV)</h1>
      <a class="btn btn-outline-secondary btn-sm" href="/breakers/admin/list.php">Back</a>
    </div>

    <?php if ($notice): ?>
      <div class="alert alert-success"><?= h($notice) ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
      <div class="alert alert-danger"><?= h($error) ?></div>
    <?php endif; ?>

    <div class="card shadow-sm">
      <div class="card-body">
        <p class="mb-2"><strong>Update/Add only.</strong> This import will not delete anything.</p>
        <ul class="small text-muted mb-3">
          <li>To remove something, use <strong>Edit → Delete Row</strong> (per-row) or set Active=0.</li>
          <li>If your CSV image links are temporary (Airtable), this importer will try to <strong>download images</strong> into /breakers/uploads/.</li>
          <li>Upsert key: uses <strong>SKU</strong> when present (your table has a UNIQUE index on SKU).</li>
        </ul>

        <form method="post" enctype="multipart/form-data">
          <input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>">
          <div class="mb-3">
            <label class="form-label">Upload CSV</label>
            <input class="form-control" type="file" name="csv" accept=".csv" required>
          </div>
          <button class="btn btn-primary" type="submit">Import</button>
          <a class="btn btn-outline-secondary" href="/breakers/admin/export.php">Export current CSV</a>
        </form>
      </div>
    </div>
  </div>
</body>
</html>
