<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

const STORES = [
  '64890' => 'Winnipeg Central',
  '83285' => 'Winnipeg South',
  '83718' => 'Winnipeg East',
];

const CACHE_DIR = __DIR__ . '/../cache';
const CACHE_TTL_SECONDS = 60 * 60; // 1 hour

if (!is_dir(CACHE_DIR)) {
  @mkdir(CACHE_DIR, 0755, true);
}

$url = isset($_GET['url']) ? trim((string)$_GET['url']) : '';
if ($url === '') {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'Missing url']);
  exit;
}

$parsed = parse_url($url);
$host = strtolower($parsed['host'] ?? '');
if ($host !== 'www.rona.ca' && $host !== 'rona.ca') {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'Only rona.ca URLs are allowed']);
  exit;
}

function build_url_with_store(string $url, string $storeId): string {
  $p = parse_url($url);
  $q = [];
  parse_str($p['query'] ?? '', $q);
  $q['viewStore'] = $storeId;

  return
    ($p['scheme'] ?? 'https') . '://' .
    ($p['host'] ?? 'www.rona.ca') .
    ($p['path'] ?? '/') .
    '?' . http_build_query($q);
}

function fetch_html(string $url, int &$httpCode, string &$err): ?string {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 10,
    CURLOPT_CONNECTTIMEOUT => 5,
    CURLOPT_USERAGENT => 'BreakerCrossRef/1.0 (+internal tool)',
    CURLOPT_HTTPHEADER => ['Accept: text/html,application/xhtml+xml'],
  ]);
  $html = curl_exec($ch);
  $err  = curl_error($ch);
  $httpCode = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);

  if ($html === false || $httpCode >= 400) return null;
  return (string)$html;
}

function parse_in_store(?string $html): array {
  if ($html === null) return ['inStore' => null, 'notAvailable' => false];
  $inStore = null;
  if (preg_match('/\b(\d+)\s+in\s+Store\b/i', $html, $m)) {
    $inStore = (int)$m[1];
  }
  $notAvailable = stripos($html, 'not currently available') !== false;
  return ['inStore' => $inStore, 'notAvailable' => $notAvailable];
}

$baseKey = sha1(($parsed['scheme'] ?? 'https') . '://' . ($parsed['host'] ?? '') . ($parsed['path'] ?? ''));

$results = [];
$anyOk = false;

foreach (STORES as $storeId => $storeName) {
  $storeUrl = build_url_with_store($url, $storeId);

  $cacheFile = CACHE_DIR . "/stock_{$baseKey}_{$storeId}.json";
  if (is_file($cacheFile)) {
    $raw = file_get_contents($cacheFile);
    if ($raw !== false) {
      $cached = json_decode($raw, true);
      if (is_array($cached) && isset($cached['ts']) && (time() - (int)$cached['ts'] < CACHE_TTL_SECONDS)) {
        $results[] = [
          'storeId' => $storeId,
          'storeName' => $storeName,
          'inStore' => $cached['inStore'] ?? null,
          'notAvailable' => $cached['notAvailable'] ?? false,
          'cached' => true,
        ];
        $anyOk = $anyOk || (bool)($cached['ok'] ?? false);
        continue;
      }
    }
  }

  $httpCode = 0; $err = '';
  $html = fetch_html($storeUrl, $httpCode, $err);

  $parsedStock = parse_in_store($html);
  $ok = $html !== null;

  $payload = [
    'ok' => $ok,
    'ts' => time(),
    'storeId' => $storeId,
    'storeName' => $storeName,
    'inStore' => $parsedStock['inStore'],
    'notAvailable' => $parsedStock['notAvailable'],
    'error' => $ok ? null : ($err !== '' ? $err : "HTTP $httpCode"),
  ];

  @file_put_contents($cacheFile, json_encode($payload));

  $results[] = [
    'storeId' => $storeId,
    'storeName' => $storeName,
    'inStore' => $payload['inStore'],
    'notAvailable' => $payload['notAvailable'],
    'cached' => false,
    'error' => $payload['error'],
  ];

  $anyOk = $anyOk || $ok;
}

echo json_encode([
  'ok' => $anyOk,
  'stores' => $results,
]);
