<?php
declare(strict_types=1);
require_once __DIR__ . '/inc/util.php';
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Breaker Cross Reference</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .app-wrap { max-width: 1200px; margin: 0 auto; }
    .result-card img { max-height: 90px; width: auto; object-fit: contain; }
    .stock-list { font-size: .95rem; }
    .mono { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace; }
  </style>
</head>
<body class="bg-light">
  <div class="container py-4 app-wrap">
    <div class="d-flex align-items-center justify-content-between mb-3">
      <h1 class="h3 m-0">Breaker Cross Reference</h1>
      <a class="btn btn-outline-secondary btn-sm" href="<?= h(build_url('admin/')) ?>">Admin</a>
    </div>

    <div class="card shadow-sm mb-3">
      <div class="card-body">
        <div class="row g-2 align-items-end">
          <div class="col-12 col-md-3">
            <label class="form-label">Brand</label>
            <select id="brand" class="form-select">
              <option value="">Any</option>
            </select>
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label">Amperage</label>
            <select id="amperage" class="form-select">
              <option value="">Any</option>
            </select>
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label">Poles</label>
            <select id="poles" class="form-select">
              <option value="">Any</option>
            </select>
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label">Safety</label>
            <select id="safety_features" class="form-select">
              <option value="">Any</option>
            </select>
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label">Mounting</label>
            <select id="mounting" class="form-select">
              <option value="">Any</option>
            </select>
          </div>
          <div class="col-12 col-md-1">
            <button id="reset" class="btn btn-outline-secondary w-100">Reset</button>
          </div>
          <div class="col-12">
            <label class="form-label">Quick search (SKU / Friendly SKU / URL / Brand)</label>
            <input id="q" class="form-control" placeholder="Type to narrow results…">
          </div>
        </div>
      </div>
    </div>

    <div id="meta" class="small text-muted mb-2"></div>
    <div id="results" class="row g-3"></div>

    <div class="text-muted small mt-4">
      <div><strong>Stock check:</strong> click “Check stock (Winnipeg)” on a result to see availability at Winnipeg Central (64890), South (83285), and East (83718).</div>
    </div>
  </div>

<script>
const els = {
  brand: document.getElementById('brand'),
  amperage: document.getElementById('amperage'),
  poles: document.getElementById('poles'),
  safety_features: document.getElementById('safety_features'),
  mounting: document.getElementById('mounting'),
  q: document.getElementById('q'),
  results: document.getElementById('results'),
  meta: document.getElementById('meta'),
  reset: document.getElementById('reset'),
};

function qs() {
  return {
    brand: els.brand.value,
    amperage: els.amperage.value,
    poles: els.poles.value,
    safety_features: els.safety_features.value,
    mounting: els.mounting.value,
    q: els.q.value.trim()
  };
}

async function loadOptions() {
  const p = new URLSearchParams(qs());
  p.delete('q'); // options don't need text search
  const res = await fetch(`<?= h(build_url('api/options.php')) ?>?${p.toString()}`);
  const data = await res.json();

  setSelect(els.brand, data.brands);
  setSelect(els.amperage, data.amperages);
  setSelect(els.poles, data.poles);
  setSelect(els.safety_features, data.safety_features);
  setSelect(els.mounting, data.mountings);
}

function setSelect(sel, values) {
  const current = sel.value;
  sel.querySelectorAll('option:not([value=""])').forEach(o => o.remove());
  (values || []).forEach(v => {
    const opt = document.createElement('option');
    opt.value = v;
    opt.textContent = v;
    sel.appendChild(opt);
  });
  // keep selection if still available
  const exists = Array.from(sel.options).some(o => o.value === current);
  if (exists) sel.value = current;
}

function card(row) {
  const img = row.image_path ? `<?= h(build_url('')) ?>${row.image_path}` : '';
  const link = row.product_link || '';
  const sku = row.sku ? String(row.sku) : '';
  const fsku = row.friendly_sku || '';

  return `
  <div class="col-12 col-md-6 col-lg-4">
    <div class="card shadow-sm h-100 result-card">
      <div class="card-body">
        <div class="d-flex gap-3">
          <div class="flex-shrink-0">
            ${img ? `<img src="${img}" alt="" class="img-fluid">` : `<div class="text-muted small">No image</div>`}
          </div>
          <div class="flex-grow-1">
            <div class="fw-semibold">${escapeHtml(row.brand || '')}</div>
            <div class="small text-muted">Bin <span class="mono">${escapeHtml(row.bin_number || '')}</span></div>
            <div class="small">
              ${row.amperage ? `<span class="badge text-bg-primary me-1">${escapeHtml(row.amperage)}</span>` : ''}
              ${row.poles ? `<span class="badge text-bg-secondary me-1">${escapeHtml(row.poles)}</span>` : ''}
              ${row.safety_features ? `<span class="badge text-bg-warning text-dark me-1">${escapeHtml(row.safety_features)}</span>` : ''}
              ${row.mounting ? `<span class="badge text-bg-info text-dark me-1">${escapeHtml(row.mounting)}</span>` : ''}
            </div>
            <div class="small mt-2">
              ${sku ? `<div><span class="text-muted">SKU:</span> <span class="mono">${escapeHtml(sku)}</span></div>` : ''}
              ${fsku ? `<div><span class="text-muted">Friendly:</span> <span class="mono">${escapeHtml(fsku)}</span></div>` : ''}
            </div>
          </div>
        </div>

        <div class="d-flex flex-wrap gap-2 mt-3">
          ${link ? `<a class="btn btn-sm btn-outline-primary" target="_blank" rel="noopener" href="${escapeAttr(link)}">View product</a>` : ''}
          ${link ? `<button class="btn btn-sm btn-outline-success check-stock" data-url="${escapeAttr(link)}">Check stock (Winnipeg)</button>` : ''}
        </div>

        <div class="stock-result mt-2 small text-muted"></div>
      </div>
    </div>
  </div>`;
}

function escapeHtml(s) {
  return String(s).replace(/[&<>"']/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]));
}
function escapeAttr(s) { return escapeHtml(s); }

async function search() {
  const p = new URLSearchParams(qs());
  const res = await fetch(`<?= h(build_url('api/search.php')) ?>?${p.toString()}`);
  const data = await res.json();
  els.meta.textContent = data.ok ? `${data.count} match(es)` : 'Error';
  els.results.innerHTML = (data.rows || []).map(card).join('') || `<div class="col-12"><div class="alert alert-warning mb-0">No matches.</div></div>`;
}

let optionsTimer = null;
function onFilterChange() {
  clearTimeout(optionsTimer);
  optionsTimer = setTimeout(async () => {
    await loadOptions();
    await search();
  }, 150);
}

['change','input'].forEach(evt => {
  els.brand.addEventListener(evt, onFilterChange);
  els.amperage.addEventListener(evt, onFilterChange);
  els.poles.addEventListener(evt, onFilterChange);
  els.safety_features.addEventListener(evt, onFilterChange);
  els.mounting.addEventListener(evt, onFilterChange);
});
els.q.addEventListener('input', () => {
  clearTimeout(optionsTimer);
  optionsTimer = setTimeout(search, 200);
});
els.reset.addEventListener('click', () => {
  els.brand.value = '';
  els.amperage.value = '';
  els.poles.value = '';
  els.safety_features.value = '';
  els.mounting.value = '';
  els.q.value = '';
  onFilterChange();
});

document.addEventListener('click', async (e) => {
  const btn = e.target.closest('.check-stock');
  if (!btn) return;

  const url = btn.dataset.url;
  const out = btn.closest('.card-body').querySelector('.stock-result');
  out.textContent = 'Checking Winnipeg stores…';

  try {
    const res = await fetch(`<?= h(build_url('api/stock.php')) ?>?url=${encodeURIComponent(url)}`);
    const data = await res.json();

    if (!data || !data.stores) {
      out.textContent = 'Stock: unavailable to verify';
      return;
    }

    const lines = data.stores.map(s => {
      if (typeof s.inStore === 'number') return `${s.storeName}: ${s.inStore} in store`;
      if (s.notAvailable) return `${s.storeName}: not available`;
      return `${s.storeName}: unavailable`;
    });

    out.innerHTML = `<div class="stock-list">${lines.map(l => `<div>• ${escapeHtml(l)}</div>`).join('')}</div>`;
  } catch (err) {
    out.textContent = 'Stock: unavailable to verify';
  }
});

(async function init() {
  await loadOptions();
  await search();
})();
</script>
</body>
</html>
